define([
    'marionette',
    'App',
    'appointmentsRouter',

    'Global',
    'Vent',
    'User',

    'HeaderView',
    'modules/page/footer/footer-view',
    'modules/landing-page/landing-page-view',
    'modules/appointments/appointments-layout-view',
    'modules/appointments/appointments-module',
    'modules/new-appointment-request/views/common/scheduling-steps/layout-view',
    'modules/feedback/feedback-view',
    'modules/new-appointment-request/view',
    'modules/notifications/notifications-router',
    'modules/new-appointment-wizard/wizard-layout-view',
    'modules/vha-enrollment/vha-enrollment-view',
    'modules/vha-register/vha-register-view',
    'GoogleAnalyticsProvider'
], function (
    Marionette,
    app,
    appointmentsRouter,

    Global,
    vent,
    User,

    HeaderView,
    FooterView,
    LandingPageView,
    AppointmentsLayoutView,
    AppointmentsModule,

    SchedulingStepsView,
    FeedbackView,
    NewAppointmentRequestLayoutView,
    notificationsRouter,
    WizardLayoutView,
    VHAEnrollmentView,
    VHARegisterView,
    GoogleAnalyticsProvider) {

    'use strict';

    var AppRouter = Marionette.AppRouter.extend({
        routes: {'*action': 'bindRoutesAndRedirect'},
        initialize: function () {
            var that = this;
            this.utilities = Global.getUtilities();

            this.globalInit();
            this.initializeVentListeners();
            this.setAnalyticsProvider(GoogleAnalyticsProvider);
        },
        globalInit: function () {
            var utilities = Global.getUtilities();

            utilities.internetConnectionCheck();
            // Check browser for compatibility
            utilities.browserAllowed();
        },
        initializeVentListeners: function () {
            var appRouter = this;

            vent.on('do:logout', function (e) {
                appRouter.logout(e);
            });

            vent.on('do:launchpad', function () {
                appRouter.launchpad();
            });

            vent.on('do:login', function () {
                appRouter.login();
            });
        },

        setAnalyticsProvider: function(provider) {
          this.analyticsProvider = provider;
        },

        bindRoutesAndRedirect: function () {
            this.bindAuthenticatedRoutes();
            Backbone.history.loadUrl();
        },

        // NOTE: app router is not started until user is logged in
        bindAuthenticatedRoutes: function () {
            if (app.patientIdentifiers.hasDfnAndIcn()) {
                this.routes = {
                    '': 'showLanding',
                    'main': 'showLanding',
                    'appointments': 'showAppointments',
                    'feedback': 'showFeedback',
                    'new-appointment-request': 'showNewAppointmentRequestWizard',
                    'new-va-appointment-request': 'showNewAppointmentRequest',
                    '*action': 'redirect',
                };
            }
            else {
                this.routes = {
                    '': 'showLanding',
                    'main': 'showLanding',
                    '*action': 'redirect',
                };
            }
            Backbone.history.handlers = [];
            this._bindRoutes();

            Backbone.history.on('route', this.sendPageview);

            if (app.patientIdentifiers.hasDfnAndIcn()) {
                this.addRoutesOn(notificationsRouter);
            }

        },

        addRoutesOn: function (subrouter) {
            this.processAppRoutes(subrouter.controller, subrouter.routes);
        },

        showHeader: function (title) {
            var header = app.mainView.getRegion('header');
            header.show(new HeaderView());
        },

        showAppointments: function () {
            app.show(new AppointmentsLayoutView({
                googleAnalyticsProvider: GoogleAnalyticsProvider,
                appointmentsModuleProvider: {
                    getInstance: function() {
                        return AppointmentsModule;
                    }
                }
            }));
        },
        showFeedback: function () {
            var content = app.mainView.getRegion('content');
            content.show(new FeedbackView());
        },

        showLanding: function () {
            var footer = app.mainView.getRegion('footer');
            this.showHeader();

            if (app.patientIdentifiers.isReady !== undefined) {
                $.when(app.patientIdentifiers.isReady).then(function () {
                    if (app.patientIdentifiers.hasDfnAndIcn()) {
                        app.mainLayoutModel.set('hasAppMenu', true);
                        this.showAppointments();
                    }
                    else if (app.patientIdentifiers.hasNoIcn()) {
                        // Has neither ICN/DFN
                        app.show(new VHAEnrollmentView());
                    }
                    else {
                        // has ICN, no DFN
                        app.mainView.model.set('hasAppMenu', false);
                        app.show(new VHARegisterView());
                    }
                    app.mainView.firstTimeShowCrisisModal();
                }.bind(this));
            }
            else {
                app.show(new LandingPageView());
            }
            footer.show(new FooterView());
        },

        showNewAppointmentRequestWizard: function () {
            app.show(new WizardLayoutView());
        },

        showNewAppointmentRequest: function () {
            app.show(new NewAppointmentRequestLayoutView({
                googleAnalyticsProvider: GoogleAnalyticsProvider,
                schedulingStepsLayoutProvider: {
                    getInstance: function(model, validator) {
                        return new SchedulingStepsView({
                            model: model,
                            validator: validator,
                        });
                    }
                }
            }));
        },

        redirect: function () {
            this.navigate('', {
                trigger: true,
                replace: true,
            });
        },
        logout: function (e) {
            e.preventDefault();
            var logoutUrl = app.resources.get('logout').get('href');
            this.utilities.cleanUpSession();
            window.location = logoutUrl + '?' + (new Date().getTime()) + '&redirect_uri=/launchpad/';
        },
        launchpad: function () {
            window.location = '../../../launchpad';
        },
        login: function () {
            this.utilities.authorize();
        },
        sendPageview: function () {
            var url = Backbone.history.getFragment();
            if (url == '') {
                if (typeof gas !== 'undefined') {
                    gas('send', 'pageview', 'appointments');
                }
            }
            else if (typeof gas !== 'undefined') {
                gas('send', 'pageview', Backbone.history.getFragment());
            }
        },
    });

    return new AppRouter();
});
